<?php

namespace App\Models;

use App\Helpers\Helper;
use App\Helpers\SaveImageTo3Path;
use App\Models\ProductAttribute;
use App\Traits\Products\Website\ProductFilterTrait;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Scopes\ProductByRollScope;


class Product extends Model
{
    use HasFactory , ProductFilterTrait;
    protected $table = 'products' ;
    private $lang ;

    protected $fillable = [
        'name_ar',
        'name_en',
        'category_id',
        'brand_id',
        'code',
        'order',
        'text_ar',
        'text_en',
        'short_text_ar',
        'short_text_en',
        'main_image',
        'second_image',
        'icon',
        'status',
        'link_ar',
        'link_en',
        'mete_title_ar',
        'mete_title_en',
        'mete_description_ar',
        'mete_description_en',
        'index',
        'approved',
        'vendor_id',
    ];

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);

        // Set the current locale dynamically
        $this->lang = Helper::getLang();

    }

    protected static function boot()
    {
        parent::boot();
        static::addGlobalScope(new ProductByRollScope );
        static::creating(function ($product) {
            if (empty($product->code)) {
                $product->code = self::generateProductCode();
            }
        });
    }

    public function delete()
    {
        SaveImageTo3Path::deleteImage($this->second_image,'products');
        SaveImageTo3Path::deleteImage($this->main_image,'products');
        SaveImageTo3Path::deleteImage($this->icon,'products');
        return parent::delete();
    }

    public static function generateProductCode()
    {
        $latestProduct = self::orderBy('id', 'desc')->first();
        $latestId = $latestProduct ? $latestProduct->id : 0;
        return config('site_perfix').date('YmdHisu');
    }
    
    public function vendor(){
        return $this->belongsTo(Vendor::class) ;
    }
    public function getVendorNameAttribute(){
        return $this->vendor?->{'name_'.$this->lang} ?? __('home.main') ;
    }


    public function getNameAttribute(){
        return $this->{'name_'.$this->lang} ;
    }

    public function getTextAttribute(){
        return $this->{'text_'.$this->lang} ;
    }

    public function getLinkAttribute(){
        return $this->{'link_'.$this->lang} ;
    }

    public function getShortTextAttribute(){
        return $this->{'short_text_'.$this->lang} ;
    }

    public function brand(){
        return $this->belongsTo(Brand::class);
    }

    public function getBrandNameAttribute(){
        return $this->brand?->name;
    }

    public function getBrandTextAttribute(){
        return $this->brand?->text;
    }

    public function getBrandLinkAttribute(){
        return $this->brand?->link;
    }

    public function category(){
        return $this->belongsTo(Category::class);
    }



    public function getCategoryNameAttribute(){
        return $this->category?->name;
    }

    public function getCategoryTextAttribute(){
        return $this->category?->text;
    }

    public function getCategoryLinkAttribute(){
        return $this->category?->link;
    }


    public function getMainImageSourceAttribute($value){

        return Helper::imageIsExists($this->main_image ,'products') ? Helper::uploadedImagesPath('products',$this->main_image)   : Helper::noImage() ;

    }

    public function getMainImage200Attribute($value){

        return Helper::imageIsExists($this->main_image ,'products') ? Helper::uploadedImages200Path('products',$this->main_image)  : Helper::noImage() ;
    }

    public function getSecondImageSourceAttribute(){

        return Helper::imageIsExists($this->second_image ,'products') ? Helper::uploadedImagesPath('products',$this->second_image)  : Helper::noImage() ;
    }

    public function getSecondImage200Attribute(){

        return Helper::imageIsExists($this->second_image ,'products') ? Helper::uploadedImages200Path('products',$this->second_image)  : Helper::noImage() ;
    }

    public function images(){

        return $this->morphMany(Image::class,'imageable'); ;
    }

    public function getActiveAttribute(){
        $this->status ? __('home.yes') : __( 'home.no') ;
     }

    public function scopeActive($query){
        $query->whereStatus(1)->whereApproved(1);
    }

    public function scopeUnactive($query){
        $query->whereStatus(0);
    }

    public function attributes(){
        return $this->hasMany(ProductAttribute::class);
    }
    public function stocks(){
        return $this->hasMany(ProductStock::class)->orderBy('stock' ,'desc');
    }

    public function getStockAttribute(){
        return $this->stocks[0] ?? null;

    }

    public function getNetPriceAttribute(){
        return $this->stock?->net_price  ;
    }
    public function getPriceAttribute(){
        return $this->stock?->price  ;
    }

    public function getstockIdAttribute(){
        return $this->stock?->id  ;
    }


    public function getDiscountPercentageAttribute(){
        return $this->stock?->product_discount_percentage ;
    }

    public function getAvailableAttribute(){
        return $this->stock?->stock  ;
    }

    public function discounts(){
        return $this->hasMany(Discount::class);
    }


    public function getDiscountIdAttribute(){
        return $this->discount?->id;
    }

    public function getDiscountStartDateAttribute(){
        return $this->discount?->start_date;
    }

    public function getDiscountEndDateAttribute(){
        return $this->discount?->end_date;
    }
    public function discount(){
        return $this->hasOne(Discount::class)
            ->where('start_date', '<=', now()->startOfDay())
            ->where('end_date', '>=', now()->startOfDay())
            ->orderByDesc('start_date') // Ensure the latest valid discount is prioritized
            ->latestOfMany();
    }

    public function getDiscountTypeAttribute(){
        return $this->discount?->type ;
    }

    public function getDiscountValueAttribute(){
        return $this->discount?->discount ;
    }

    public function getAttributeValuesAttribute(){
        $product_Attributes_id = $this->attributes()->pluck('id')->ToArray();
        return ProductAttributeValue::whereIn('product_attribute_id' , $product_Attributes_id)->get();
    }

    public function scopeHasStock($query){
        return $query->whereHas('stocks');
    }

    public function scopeLink($query ,$link){
        return $query->where('link_ar',$link)->orWhere('link_en' ,$link);
    }


    public function reviews(){
        return $this->hasMany(ProductReview::class);
    }

    public function getReviewsCountAttribute()
    {
        return (float) $this->reviews->count();
    }

    public function getReviewsPercentageAttribute()
    {
        return (float) number_format(($this->reviews->avg('rate') / 5) * 100, 2, '.', '');
    }

    public function getReviewsAvrageAttribute()
    {
        return (float) number_format($this->reviews->avg('rate'), 2, '.', '');
    }

    public function getReviewsRate5CountAttribute()
    {
        return (float) number_format($this->reviews()->where('rate', 5)->count(), 2, '.', '');
    }

    public function getReviewsRate4CountAttribute()
    {
        return (float) number_format($this->reviews()->where('rate', 4)->count(), 2, '.', '');
    }

    public function getReviewsRate3CountAttribute()
    {
        return (float) number_format($this->reviews()->where('rate', 3)->count(), 2, '.', '');
    }

    public function getReviewsRate2CountAttribute()
    {
        return (float) number_format($this->reviews()->where('rate', 2)->count(), 2, '.', '');
    }

    public function getReviewsRate1CountAttribute()
    {
        return (float) number_format($this->reviews()->where('rate', 1)->count(), 2, '.', '');
    }


    public function scopeOrders($q){
        return $q->whereHas('stocks' ,function($q) {
            $q->whereHas('orders');
        });
    }

    public function scopeStockNotInWarehouse($q){
        return $q->whereHas('stocks', function($q){
            $q->where('stock', '=', 0);
        });
    }

    public function scopeStockInWarehouse($q){
        return $q->whereHas('stocks', function($q){
            $q->where('stock', '>', 0);
        });
    }


    public function scopeHasDiscount($q){
        return $q->whereHas('discount');
    }

}
