<?php

namespace App\Http\Controllers\Website;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\OrderGuestRequest;
use App\Models\Address;
use App\Models\Area;
use App\Models\Cart;
use App\Models\Country;
use App\Models\Order;
use App\Models\Region;
use App\Models\User;
use App\Traits\OrderTrait;
use App\Traits\RegisterTriat;
use App\Traits\SeoTrait;
use App\Traits\ShippingTrait;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

use Illuminate\Support\Facades\Mail;
use App\Mails\OrderConfirmedMail ;

class OrderController extends Controller
{

    use SeoTrait ,ShippingTrait ,RegisterTriat, OrderTrait ;

    public function index()
    {
        if(auth()->check()){
            $data =  $this->checkoutAuth();
            return view('website.checkout.auth.index',$data);
        }

        $data =  $this->checkoutGuest();
        return view('website.checkout.guest.index',$data);
    }

    public function placeOrder(OrderGuestRequest $request)
    {
        $requestValedated = $request->validated() ;
        // get auth user or create new user
        $user = auth()->check() ? auth()->user() : $this->createNewUser( $requestValedated) ;

        try {
            DB::beginTransaction();
            // check items stock is if already have and assign requier item for for order product
            [$items_status , $cart] = $this->checkProduct() ;
            // assign user id to data validated array ;
            $requestValedated['user_id'] = $user->id ;
            // update user address if has and chenged or create new address for current user
            $address = Address::updateOrCreate(['id' =>  $requestValedated['address_id']?? null ],$requestValedated) ;

            //  if an item stock less than order quantity we will error has details
            if(!$items_status){
                return redirect()->back() ;
            }

            $orderData = $this-> getOrderData($cart, $address ,$requestValedated, $user) ;
            $order = Order::create($orderData) ;
            $order->orderStatus()->create(['status' => 'pending'])  ;
            $this->saveOrderItems($cart ,$order) ;
            if($cart->coupon){
                $cart->coupon->users()->create(['user_id' => $user->id]) ;
            }
            DB::commit() ;
            Cart::find($cart->id)->delete() ;
            try{
                Mail::to($user->email)->queue(new OrderConfirmedMail($order));
            }catch(Exception $th){
                Helper::messageError($th->getMessage().' an error occurred when send email' );
            }

        } catch (Exception $th) {
            DB::rollBack() ;
            Helper::messageError( __('home.an error occurred'));
            return redirect()->back() ;

        }
        // $this->notifyAdmins() ;
        $lang = Helper::getLang();
       return redirect($lang.'/thanks/'.$order->number) ;

    }

    public function getShippingFees()
    {
        $dataValidated = request()->validate([
            'area_id' => 'required|exists:areas,id'
        ]);
        return response()->json(['shipping_fees' => $this->getShipping(  $dataValidated['area_id']) ]) ;
    }

    private function saveOrderItems($cart ,$order){
        foreach ($cart->items as $key => $item) {
            $order->products()->create($item->toArray()) ;
            $current_stock = $item->product_current_stock ;
            $new_stock = $current_stock - $item->quantity ;
            $item->productStock()->update(['stock' => $new_stock]);
        }
    }

    private function notifyAdmins()
    {
        try {
            $admins = User::role('admin')->get();
                foreach ( $admins as $key => $value) {
                    Helper::SendEmail($value->email ,"there is new order for ".auth()->user()->name ,env('APP_NAME')) ;
                }

        } catch (Exception $e) {

        }
    }



    private function checkoutAuth(){
        $data['user'] = auth()->user() ;
        $data['shipping_fees'] = $this->getShipping() ;
        $data['countries'] = Country::all();
        $data['regions'] = Region::byCountryAuth()->get();
        $data['areas'] = Area::byRegionAuth()->get();

        return $data;

    }

    private function checkoutGuest(){
        $data['shipping_fees'] = $this->getShipping() ;
        $data['countries'] = Country::all();
        return $data;

    }

    private function getOrderData(...$data){
        [$cart, $address, $requestValidated ,$user] = $data;
        $shipping_feeas = $this->getShipping() ;
        // Merge additional data into the existing cart object
        $cartData = array_merge($cart->toArray(), [
            'products_price' => $cart->total_price,
            'total_price' =>  Helper::checkShippingFree($cart->net_total_price,$shipping_feeas) ,
            'shipping_fees' =>  Helper::checkShippingFree($cart->net_total_price,$shipping_feeas) - $cart->net_total_price ,
            'coupon_discount' => $cart->coupon_amount ,
            'note' => $requestValidated['note'] ?? '',
            'shipping_method' => config('site_shipping_status'),
            'address_id' => $address->id ,
            'user_id' => $cart->user_id ,
        ]);

        return $cartData;
    }



}
